/*

Founts by David Ashley
http://www.xdr.com/dash
dash@xdr.com
Written about May 1999

Inspired by Eric's Cascade which is overpriced at $39. We needed a free
equivalent!

This is free software released under the terms of the GPL

*/




#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <fcntl.h>

#include "SDL.h"

#include "founts.h"


#define PI 3.1415926535897932

#define VIRTX 640
#define VIRTY 480

int IXSIZE=VIRTX;
int IYSIZE=VIRTY;

#define NOTDOWN 0
#define JUSTDOWN 1
#define CONTDOWN 2

#define NUMTYPES 14


#define FRACTION 8
#define SPEEDDELTA (1<<FRACTION-1)
#define SPEEDMAX (10<<FRACTION)

#define BOTTOM 16

#define MAXBALLS 4096
#define MAXHOSES 64
#define MAXBLOCKS 64



#define MAXCOLORS 256

unsigned short *dummyball,*colorballs[32],*ballarray[512];

int gravity=8;

#define MIXFRAC 3
int vals[4][(VIRTY>>MIXFRAC)*(VIRTX>>MIXFRAC)];

unsigned char *block64;

char pauseflag=0;

#define REGISTERLEN (1+4+4+4+16+1)

char mixing,dospiral,dosphere,dohide,rotating;

int spinangle;

int mousex,mousey;
int buttonstate;

char sdlloaded=0;

unsigned char *bbm;

SDL_Surface *thescreen;
SDL_Color themap[256];
unsigned char mymap[3*MAXCOLORS];
int allocatedcolors=1;
int usedcolors=1;
int stridemult[2048];
int bitsperpixel;

unsigned char redvals[]=  {255,0,0,255,0,255,255,0,64,96,128,160,192,224};
unsigned char greenvals[]={0,255,0,255,255,0,255,0,0,0,0,0,0,0};
unsigned char bluevals[]= {0,0,255,0,255,255,255,128,0,0,0,0,0,0};

char *preset[10]={0};

unsigned char *fractback=0;


void *things=0;
int thingsize,thingnum;

int dopcx(char *name,gfxset *gs);

int hc=0;
int gamemode;
char exitflag=0;
int framecount=0;

unsigned char regpacket[64];
char gameversion[4]={0xda,0x01,0x00,0x01};
char playername[16];

int stride;
unsigned char *videomem;
int frames;

char havepulse=0;

struct point
{
int x,y;
};
struct fpoint
{
float x,y;
};

struct ball
{
	struct ball *next;
	int x,y;
	int dx,dy;
	char flags;
	unsigned char red,green,blue;
	int index;
} *balls,*firstball,*activeballs;


struct hose
{
	struct hose *next;
	int x,y;
	int angle;
	unsigned char red,green,blue;
	int index;
} *hoses,*firsthose,*activehoses,fakehose;

struct block
{
	struct block *next;
	int x,y;
	int angle;
} *blocks,*firstblock,*activeblocks,fakeblock;


#define TAP1 250
#define TAP2 103
/*
#define TAP1 55
#define TAP2 31
*/

int maprgb(int red,int green,int blue)
{
	if(bitsperpixel==16)
		return SDL_MapRGB(thescreen->format,red,green,blue);
	red&=0xe0;
	green&=0xe0;
	blue&=0xc0;
	return (red>>5) | (green>>2) | blue;
}

unsigned char myrandblock[TAP1];
int myrandtake;

unsigned short *makeball(int radius,uchar red,uchar green,uchar blue)
{
unsigned short *temp,*put;
int i,j,k,c;
unsigned char *p;
struct figure *afig;
float lx,ly,lz,mag,x,y,z,y2,radius2,intensity,t;
unsigned char intens[4096];

	radius=radius*IXSIZE/640;
	if(radius<2) radius=2;
	radius2=radius*radius;
	lx=-1.0;
	ly=-1.0;
	lz=1.5;
	mag=sqrt(lx*lx+ly*ly+lz*lz)*radius;
	lx/=mag;
	ly/=mag;
	lz/=mag;

	put=temp=(unsigned short *)block64;
	*put++=-radius;
	*put++=-radius;
	*put++=radius+radius+1;
	*put++=radius+radius+1;
	memset(intens,0,sizeof(intens));
	for(j=0;j<radius+radius;++j)
	{
		y=j-radius+0.5;
		y2=y*y;
		p=intens+(j<<6);
		for(i=0;i<radius+radius;++i,++p)
		{
			x=i-radius+0.5;
			t=x*x+y2;
			if(t>radius2) continue;
			z=sqrt(radius2-t);
			intensity=(x*lx+y*ly+z*lz+1.0)/2.0;
			intensity*=intensity;
			*p=intensity*255;
		}
	}
	for(j=0;j<=radius+radius;++j)
	{
		p=intens+(j<<6);
		c=0;
		i=0;
		while(!*p && i<60)
		{
			++p;
			++i;
		}
		if(i>=60) continue;
		while(c<=radius+radius && p[c])
			++c;
		*put++=c;
		*put++=i;
		*put++=j;
		while(c--)
		{
			k=*p++;
			*put++=maprgb(red*k>>8,green*k>>8,blue*k>>8);
		}
	}
	*put++=0;
	i=put-temp<<1;
	put=malloc(i);
	if(put)
		memcpy(put,temp,i);
	return put;
}


myrand1()
{
int i;
int val;

	i=myrandtake-TAP2;
	if(i<0) i+=TAP1;
	val=myrandblock[myrandtake++]^=myrandblock[i];
	if(myrandtake==TAP1) myrandtake=0;
	return val;
}
myrand()
{
int v;
	v=myrand1();
	return (v<<8) | myrand1();
}

initmyrand()
{
int i,j;
char *p;
int msb,msk;

	myrandtake=0;
	p=myrandblock;
	j=12345;
	i=TAP1;
	while(i--)
	{
		j=(j*1277)&0xffff;
		*p++=j>>8;
	}
	p=myrandblock+14;
	msk=0xff;
	msb=0x80;
	do
	{
		*p&=msk;
		*p|=msb;
		p+=11;
		msk>>=1;
		msb>>=1;
	} while(msk);
	i=500;
	while(i--) myrand();

}



long longind(unsigned char *p)
{
	return (p[0]<<24L) | (p[1]<<16L) | (p[2]<<8) | p[3];
}
short shortind(unsigned char *p)
{
	return (p[0]<<8L) | p[1];
}


nomem(char *str)
{
	printf("No memory!!![%s]\n",str);
	closex();
	exit(1);
}


vsync()
{
	scrunlock();
	scrlock();
}

int dopcx(char *name,gfxset *gs)
{
int err;
	err=dopcxreal(name,gs);
	if(err)
		printf("Error loading \"%s\":code %d\n",name,err);
	return err;
}

#define IBUFFLEN 1024
int ileft=0,ihand=0,byteswide;
unsigned char ibuff[IBUFFLEN],*itake;

int myci()
{
	if(!ileft)
	{
		ileft=read(ihand,ibuff,IBUFFLEN);

		if(!ileft) return -1;
		itake=ibuff;
	}
	ileft--;
	return *itake++;
}


int dopcxreal(char *name,gfxset *gs)
{
int xs,ys;
int i,j,k;
int totalsize;
int width,height;
unsigned char *bm,*lp;
char tname[256];


	memset(gs,0,sizeof(gfxset));
	ileft=0;
	sprintf(tname,"data/%s",name);
	ihand=open(tname,O_RDONLY);
	if(ihand<0)
	{
		char tname2[256];
		sprintf(tname2,"%s.pcx",tname);
		ihand=open(tname2,O_RDONLY);
		if(ihand<0)
			return 1;
	}
	if(myci()!=10) {close(ihand);return 2;} // 10=zsoft .pcx
	if(myci()!=5) {close(ihand);return 3;} // version 3.0
	if(myci()!=1) {close(ihand);return 4;} //encoding method
	if(myci()!=8) {close(ihand);return 5;} //bpp
	xs=myci();
	xs|=myci()<<8;
	ys=myci();
	ys|=myci()<<8;
	width=myci();
	width|=myci()<<8;
	height=myci();
	height|=myci()<<8;
	width=width+1-xs;
	height=height+1-ys;
	for(i=0;i<48+4;++i) myci();
	myci();
	if(myci()!=1) {close(ihand);return 6;} // # of planes
	byteswide=myci();
	byteswide|=myci()<<8;
	i=myci();
	i|=myci()<<8;
//	if(i!=1) {close(ihand);return 7;} // 1=color/bw,2=grey
	for(i=0;i<58;++i) myci();
	totalsize=height*byteswide;
	bm=malloc(totalsize+1);
	if(!bm) {close(ihand);return 8;} // no memory
	gs->gs_pic=bm;
	gs->gs_xsize=width;
	gs->gs_ysize=height;
	while(height--)
	{
		lp=bm;
		i=byteswide;
		while(i>0)
		{
			j=myci();
			if(j<0xc0)
			{
				*lp++=j;
				--i;
			} else
			{
				j&=0x3f;
				k=myci();
				while(j-- && i)
				{
					*lp++=k;
					--i;
				}
			}
		}
		bm+=width;
	}
	lseek(ihand,-0x300,SEEK_END);
	read(ihand,gs->gs_colormap,0x300);
	close(ihand);
	return 0;
	
}

#define SGN(x) ((x)==0 ? 0 : ((x)<0 ? -1 : 1))

void *allocentry()
{
list *entry;
	if(!(entry=((list *)things)->next)) return 0;
	((list *)things)->next=entry->next;
	memset(entry,0,thingsize);
	return entry;
}
void freeentry(void *entry)
{
	((list *)entry)->next=((list *)things)->next;
	((list *)things)->next=entry;
}
void addtail(void *header,void *entry)
{
	while(((list *)header)->next) ((list *)header)=((list *)header)->next;
	((list *)header)->next=entry;
	((list *)entry)->next=0;
}
void delink(void *header,void *entry)
{
	while(((list *)header)->next != entry) header=((list *)header)->next;
	((list *)header)->next=((list *)entry)->next;
	((list *)entry)->next=0;
	freeentry(entry);
}
void initlist(void *first,int size,int num)
{
unsigned char *p;
list *usable;
int i;

	memset(first,0,size*num);
	for(i=0;i<num-1;++i)
	{
		p=first;
		p+=size;
		((list *)first)->next=(void *)p;
		first=p;
	}
	((list *)first)->next=0;
}

void initheader(void *p)
{
	memset(p,0,sizeof(list));
}

char interrupted;

Uint32 sdlhandler(Uint32 time)
{
	interrupted=1;
	hc++;
	return time;
}
pulseon()
{
	if(havepulse) return;
	havepulse=1;
	SDL_SetTimer(10,sdlhandler);
	hc=interrupted=0;
}
pulseoff()
{
	if(!havepulse) return;
	havepulse=0;
	SDL_SetTimer(0,0);
	
}

initballs()
{
int i;
	balls=malloc(MAXBALLS*sizeof(struct ball));
	if(!balls) nomem("initballs");
	for(i=0;i<MAXBALLS-1;++i)
		balls[i].next=balls+i+1;
	balls[i].next=0;
	firstball=balls;
	activeballs=0;
}
inithoses()
{
int i;
	hoses=malloc(MAXHOSES*sizeof(struct hose));
	if(!hoses) nomem("inithoses");
	for(i=0;i<MAXHOSES-1;++i)
		hoses[i].next=hoses+i+1;
	hoses[i].next=0;
	firsthose=hoses;
	activehoses=0;
	fakehose.x=VIRTX<<FRACTION-4;
	fakehose.y=VIRTY<<FRACTION-4;
	fakehose.angle=384;
	fakehose.red=fakehose.green=fakehose.blue=255;
	fakehose.index=511;
}
initblocks()
{
int i;
	blocks=malloc(MAXBLOCKS*sizeof(struct block));
	if(!blocks) nomem("initblocks");
	for(i=0;i<MAXBLOCKS-1;++i)
		blocks[i].next=blocks+i+1;
	blocks[i].next=0;
	firstblock=blocks;
	activeblocks=0;
	fakeblock.x=4*VIRTX<<FRACTION-4;
	fakeblock.y=VIRTY<<FRACTION-4;
	fakeblock.angle=0;
}
void scanfunc(int x1,int y,int x2,long color)
{
unsigned short *p1;
unsigned char *p;
int t;

	if(x1>x2)
	{
		t=x1;
		x1=x2;
		x2=t;
	}
	if(y<0 || y>=IYSIZE || x1>=IXSIZE || x2<0) return;
	if(x1<0) x1=0;
	if(x2>=IXSIZE) x2=IXSIZE-1;
	x2-=x1;
	if(bitsperpixel==16)
	{
		p1=(unsigned short *)(videomem+stride*y+(x1<<1));
		while(x2-->0)
			*p1++=color;
	} else
	{
		p=videomem+stride*y+x1;
		while(x2-->0)
			*p++=color;
	}
}

struct ball *allocball()
{
struct ball *aball;
	if(!firstball) return 0;
	aball=firstball;
	firstball=firstball->next;
	memset(aball,0,sizeof(struct ball));
	return aball;
}
freeball(struct ball *aball)
{
	aball->next=firstball;
	firstball=aball;
}
struct hose *allochose()
{
struct hose *ahose;
	if(!firsthose) return 0;
	ahose=firsthose;
	firsthose=firsthose->next;
	memset(ahose,0,sizeof(struct hose));
	return ahose;
}
freehose(struct hose *ahose)
{
	ahose->next=firsthose;
	firsthose=ahose;
}
struct block *allocblock()
{
struct block *ablock;
	if(!firstblock) return 0;
	ablock=firstblock;
	firstblock=firstblock->next;
	memset(ablock,0,sizeof(struct block));
	return ablock;
}
freeblock(struct block *ablock)
{
	ablock->next=firstblock;
	firstblock=ablock;
}

#define VEL 2
#define HOSESIZE 30
#define BLOCKSIZE 12

dohoses()
{
struct hose *ahose;
struct ball *aball;
float angle;
int dx,dy;
int side;

	ahose=activehoses;
	while(ahose)
	{
		aball=allocball();
		if(aball)
		{
			angle=(ahose->angle+spinangle)*PI/512;
			dx=-(1<<FRACTION)*cos(angle);
			dy=-(1<<FRACTION)*sin(angle);
			aball->x=ahose->x+dx*HOSESIZE;
			aball->y=ahose->y+dy*HOSESIZE;
			side=((rand()&255)-127)*HOSESIZE/3;
			dx=dx*side>>7;
			dy=dy*side>>7;
			aball->x-=dy;
			aball->y+=dx;

			aball->dx=-(VEL<<FRACTION)*cos(angle);
			aball->dy=-(VEL<<FRACTION)*sin(angle);
			aball->red=ahose->red;
			aball->green=ahose->green;
			aball->blue=ahose->blue;
			aball->index=ahose->index;
			aball->next=activeballs;
			activeballs=aball;
		}
		ahose=ahose->next;
	}
}

hosecontrol1(struct point *p,struct hose *ahose)
{
	p->x=ahose->x>>FRACTION;
	p->y=ahose->y>>FRACTION;
}
hosecontrol2(struct point *p,struct hose *ahose)
{
float a;
int x,y;

	a=(ahose->angle+spinangle)*PI/512;
	x=ahose->x>>FRACTION;
	y=ahose->y>>FRACTION;
	p->x=x-HOSESIZE*3/4*cos(a);
	p->y=y-HOSESIZE*3/4*sin(a);
}
int near(int v1,int v2)
{
	v1-=v2;
	if(v1<0) v1=-v1;
	return v1<8;
}
int nearrad(int dx,int dy,int rad)
{
	return dx*dx+dy*dy<rad*rad;
}
makeindex(int red,int green,int blue)
{
	return ((red&0xe0)<<1) | ((green&0xe0)>>2) | ((blue&0xe0)>>5);
}

struct hose *addhose(int red,int green,int blue)
{
struct hose *ahose;
	ahose=allochose();
	if(!ahose) return;
	ahose->next=activehoses;
	activehoses=ahose;
	ahose->angle=384;
	ahose->x=VIRTX<<FRACTION-1;
	ahose->y=VIRTY<<FRACTION-1;
	ahose->red=red;
	ahose->green=green;
	ahose->blue=blue;
	ahose->index=makeindex(red,green,blue);
	return ahose;
}
struct block *addblock()
{
struct block *ablock;
	ablock=allocblock();
	ablock->next=activeblocks;
	activeblocks=ablock;
	ablock->angle=0;
	ablock->x=VIRTX<<FRACTION-1;
	ablock->y=VIRTY<<FRACTION-1;
	return ablock;
}


int checkhose(struct hose *ahose)
{
struct point p;
	hosecontrol1(&p,ahose);
	if(near(p.x,mousex) && near(p.y,mousey))
		return 1;
	hosecontrol2(&p,ahose);
	if(near(p.x,mousex) && near(p.y,mousey))
		return 2;
	return 0;
}
struct point colorpoints[NUMTYPES];
int colorindexes[NUMTYPES];
unsigned short *colorfigs[NUMTYPES];
initcolors()
{
int i,j,k,r,g,b;
	for(i=0;i<NUMTYPES;++i)
	{
		colorpoints[i].y=15*VIRTY/16;
		colorpoints[i].x=(i+1)*VIRTX/25;
		j=(i&7)<<5;
		k=i>>3;
		r=redvals[i];
		g=greenvals[i];
		b=bluevals[i];
		colorindexes[i]=makeindex(r,g,b);
		colorfigs[i]=makeball(12,r,g,b);
	}
}


hosehit()
{
static struct hose *ahose;
static int controltype;
struct point p;
float x,y,ax,ay,d,angle;
int i,j;

	if(buttonstate==NOTDOWN) return;

	if(buttonstate==JUSTDOWN)
	{
		ahose=activehoses;
		controltype=0;
		while(ahose)
		{
			if(controltype=checkhose(ahose))
				break;
			ahose=ahose->next;
		}
		if(!ahose && (controltype=checkhose(&fakehose)))
		{
			if(controltype==1)
				ahose=addhose(255,255,255);
			else
				controltype=0;
		}
	}
	if(controltype)
	{
		if(!ahose) controltype=0;
		switch(controltype)
		{
		case 1:
			ahose->x=mousex<<FRACTION;
			ahose->y=mousey<<FRACTION;
			for(i=0;i<NUMTYPES;++i)
			{
				if(nearrad(mousex-colorpoints[i].x,mousey-colorpoints[i].y,12))
				{
					ahose->red=redvals[i];
					ahose->green=greenvals[i];
					ahose->blue=bluevals[i];

					ahose->index=colorindexes[i];
					break;
				}
			}
			break;
		case 2:
			x=(ahose->x>>FRACTION)-mousex;
			y=(ahose->y>>FRACTION)-mousey;
			if(!x && !y) break;
			d=sqrt(x*x+y*y);
			x/=d;
			y/=d;
			ax=x<0 ? -x : x;
			ay=y<0 ? -y : y;
			if(ax>ay)
			{
				angle=acos(x);
				if(y<0) angle=-angle;
			} else
			{
				angle=asin(y);
				if(x<0) angle=PI-angle;
			}
			angle*=512/PI;
			ahose->angle=angle;
			break;
		}
	}
}
blockhit()
{
static struct block *ablock;
static int controltype;
struct point p;
float x,y,ax,ay,d,angle;
int i;

	if(buttonstate==NOTDOWN) return;

	if(buttonstate==JUSTDOWN)
	{
		ablock=activeblocks;
		controltype=0;
		while(ablock)
		{
			if(controltype=checkblock(ablock))
				break;
			ablock=ablock->next;
		}
		if(!ablock && (controltype=checkblock(&fakeblock)))
		{
			if(controltype==1)
				ablock=addblock();
			else
				controltype=0;
		}
	}
	if(controltype)
	{
		if(!ablock) controltype=0;
		switch(controltype)
		{
		case 1:
			ablock->x=mousex<<FRACTION;
			ablock->y=mousey<<FRACTION;
			for(i=0;i<NUMTYPES;++i)
			{
				if(near(mousex,colorpoints[i].x) &&
					near(mousey,colorpoints[i].y));
			}
			makebbm();
			break;
		case 2:
			x=mousex-(ablock->x>>FRACTION);
			y=mousey-(ablock->y>>FRACTION);
			if(!x && !y) break;
			d=sqrt(x*x+y*y);
			x/=d;
			y/=d;
			ax=x<0 ? -x : x;
			ay=y<0 ? -y : y;
			if(ax>ay)
			{
				angle=acos(x);
				if(y<0) angle=-angle;
			} else
			{
				angle=asin(y);
				if(x<0) angle=PI-angle;
			}
			angle*=512/PI;
			ablock->angle=angle;
			makebbm();
			break;
		}
	}
}

#define HOSERAD ((float)HOSESIZE*IXSIZE/VIRTX)

plothose(struct hose *ahose)
{
int x1,y1,x2,y2,x3,y3,x,y;
float a;
float r;
struct point p;
int angle;
	angle=ahose->angle+spinangle;
	x=(float)ahose->x*IXSIZE/VIRTX;
	y=(float)ahose->y*IYSIZE/VIRTY;
	x>>=FRACTION;
	y>>=FRACTION;
	a=angle*PI/512;
	x1=x+HOSERAD*cos(a);
	y1=y+HOSERAD*sin(a);
	a+=150*PI/180;
	x2=x+HOSERAD*cos(a);
	y2=y+HOSERAD*sin(a);
	a+=60*PI/180;
	x3=x+HOSERAD*cos(a);
	y3=y+HOSERAD*sin(a);
	tri(x1,y1,x2,y2,x3,y3,scanfunc,maprgb(ahose->red,ahose->green,ahose->blue));
	drawfigure(x,y,dummyball);
	hosecontrol2(&p,ahose);
	drawfigure(p.x*IXSIZE/VIRTX,p.y*IYSIZE/VIRTY,dummyball);
}

plothoses()
{
struct hose *ahose;

	ahose=activehoses;
	while(ahose)
	{
		plothose(ahose);
		ahose=ahose->next;
	}
	plothose(&fakehose);
}
#define BLOCKRAD ((float)BLOCKSIZE*IXSIZE/VIRTX)

blockcontrol1(struct point *p,struct block *ablock)
{
	p->x=ablock->x>>FRACTION;
	p->y=ablock->y>>FRACTION;
}
blockcontrol2(struct point *p,struct block *ablock)
{
float a;
int x,y;

	a=ablock->angle*PI/512;
	x=ablock->x>>FRACTION;
	y=ablock->y>>FRACTION;
	p->x=x+BLOCKSIZE*3*cos(a);
	p->y=y+BLOCKSIZE*3*sin(a);
}
int checkblock(struct block *ablock)
{
struct point p;
	blockcontrol1(&p,ablock);
	if(near(p.x,mousex) && near(p.y,mousey))
		return 1;
	blockcontrol2(&p,ablock);
	if(near(p.x,mousex) && near(p.y,mousey))
		return 2;
	return 0;
}
blockpoint1(struct fpoint *fp,struct block *ablock)
{
float a;
	a=ablock->angle*PI/512;
	fp->x=BLOCKSIZE*cos(a);
	fp->y=BLOCKSIZE*sin(a);
}
blockpoint2(struct point *p,struct block *ablock)
{
float a;
	a=(ablock->angle-40)*PI/512;
	p->x=BLOCKRAD*cos(a);
	p->y=BLOCKRAD*sin(a);
}
#define BLOCKFACTOR 4
plotblock(struct block *ablock)
{
int x1,y1,x2,y2,x3,y3,x4,y4,x5,y5,x6,y6,x,y;
float a;
float r;
struct point p;
float dx,dy;
struct fpoint fp;


	x=(float)ablock->x*IXSIZE/VIRTX;
	y=(float)ablock->y*IYSIZE/VIRTY;
	x>>=FRACTION;
	y>>=FRACTION;
	blockpoint1(&fp,ablock);
	dx=fp.x*IXSIZE/VIRTX;
	dy=fp.y*IYSIZE/VIRTY;

	x1=x+BLOCKFACTOR*dx+.5;
	y1=y+BLOCKFACTOR*dy+.5;
	x2=x-BLOCKFACTOR*dx+.5;
	y2=y-BLOCKFACTOR*dy+.5;
	x3=x1+dx-dy+.5;
	y3=y1+dy+dx+.5;
	x4=x1+dx+dy+.5;
	y4=y1+dy-dx+.5;
	x5=x2-dx-dy+.5;
	y5=y2-dy+dx+.5;
	x6=x2-dx+dy+.5;
	y6=y2-dy-dx+.5;

	tri(x1,y1,x3,y3,x4,y4,scanfunc,maprgb(96,96,96));
	tri(x2,y2,x5,y5,x6,y6,scanfunc,maprgb(96,96,96));

	tri(x1,y1,x2,y2,x6,y6,scanfunc,maprgb(96,96,96));
	tri(x1,y1,x4,y4,x6,y6,scanfunc,maprgb(96,96,96));

	tri(x1,y1,x2,y2,x5,y5,scanfunc,maprgb(96,96,96));
	tri(x1,y1,x3,y3,x5,y5,scanfunc,maprgb(96,96,96));

	drawfigure(x,y,dummyball);
	blockcontrol2(&p,ablock);
	drawfigure(p.x*IXSIZE/VIRTX,p.y*IYSIZE/VIRTY,dummyball);
}
initbbm()
{
	bbm=malloc(VIRTX*VIRTY);
	if(!bbm)
		nomem("initbitmap");
	clearbbm();
}
clearbbm()
{
	memset(bbm,0,VIRTX*VIRTY);
}
void bbmscan(int x1,int y,int x2,long color)
{
unsigned char *p1;
int t;

	if(x1>x2)
	{
		t=x1;
		x1=x2;
		x2=t;
	}
	if(y<0 || y>=VIRTY || x1>=VIRTX || x2<0) return;
	if(x1<0) x1=0;
	if(x2>=VIRTX) x2=VIRTX-1;
	p1=bbm+VIRTX*y+x1;
	if(x2-x1)
		memset(p1,color,x2-x1);
}

float blockangles[256];

makebbm()
{
struct block *ablock;
int x1,y1,x2,y2,x3,y3,x4,y4,x5,y5,x6,y6,x,y;
float dx,dy;
struct fpoint fp;
int id;

	ablock=activeblocks;
	clearbbm();
	id=1;
	while(ablock)
	{
		x=ablock->x>>FRACTION;
		y=ablock->y>>FRACTION;


		x=(float)ablock->x;
		y=(float)ablock->y;
		x>>=FRACTION;
		y>>=FRACTION;
		blockpoint1(&fp,ablock);
		dx=fp.x;
		dy=fp.y;

		x1=x+BLOCKFACTOR*dx+.5;
		y1=y+BLOCKFACTOR*dy+.5;
		x2=x-BLOCKFACTOR*dx+.5;
		y2=y-BLOCKFACTOR*dy+.5;
		x3=x1+dx-dy+.5;
		y3=y1+dy+dx+.5;
		x4=x1+dx+dy+.5;
		y4=y1+dy-dx+.5;
		x5=x2-dx-dy+.5;
		y5=y2-dy+dx+.5;
		x6=x2-dx+dy+.5;
		y6=y2-dy-dx+.5;

		tri(x1,y1,x3,y3,x4,y4,bbmscan,id);
		blockangles[id]=ablock->angle&1023;
		++id;
		tri(x2,y2,x5,y5,x6,y6,bbmscan,id);
		blockangles[id]=ablock->angle-512&1023;
		++id;
		tri(x1,y1,x2,y2,x6,y6,bbmscan,id);
		tri(x1,y1,x4,y4,x6,y6,bbmscan,id);
		blockangles[id]=ablock->angle-256&1023;
		++id;
		tri(x1,y1,x2,y2,x5,y5,bbmscan,id);
		tri(x1,y1,x3,y3,x5,y5,bbmscan,id);
		blockangles[id]=ablock->angle+256&1023;
		++id;
		ablock=ablock->next;
	}
}
#define ELAST 1.7

bbmball(struct ball *aball)
{
int x,y;
int id;
float dx,dy,v,s,c,a;

	aball->x+=aball->dx;
	aball->y+=aball->dy;
	x=aball->x>>FRACTION;
	y=aball->y>>FRACTION;

	if(x<0 || x>=VIRTX || y<0 || y>=VIRTY) return;
	if(!(id=bbm[y*VIRTX+x])) return;
	a=blockangles[id]*PI/512;
	s=sin(a);
	c=cos(a);
	dx=aball->dx;
	dy=aball->dy;
	v=dx*c+dy*s;
	if(v<0)
	{
		dx=c*v;
		dy=s*v;
		aball->dx-=dx*ELAST;
		aball->dy-=dy*ELAST;
	}

}



plotblocks()
{
struct block *ablock;

	ablock=activeblocks;
	while(ablock)
	{
		plotblock(ablock);
		ablock=ablock->next;
	}
	plotblock(&fakeblock);
}

drawcolors()
{
int x,y;
int dx,dy;
int i,j,k;
int r,g,b;
unsigned short *afig;

	dx=6*IXSIZE/VIRTX;
	dy=6*IYSIZE/VIRTY;

	for(i=0;i<NUMTYPES;++i)
	{
		x=colorpoints[i].x*IXSIZE/VIRTX;
		y=colorpoints[i].y*IYSIZE/VIRTY;
		drawfigure(x-dx,y-dy,colorfigs[i]);
	}
}

char disk[]={
-1,-4,3,
-3,-3,7,
-3,-2,7,
-4,-1,9,
-4, 0,9,
-4, 1,9,
-3, 2,7,
-3, 3,7,
-1, 4,3,
128
};

drawsolid(int x,int y,char *p,uchar red,uchar green,uchar blue)
{
int tx,ty,tl;
unsigned short *put,color;
unsigned char *pc;

	color=maprgb(red,green,blue);
	while(*p!=-128)
	{
		tx=x+*p++;
		ty=y+*p++;
		tl=*p++;
		if(ty<0 || ty>=IYSIZE) continue;
		if(tx+tl<0 || tx>=IXSIZE) continue;
		if(tx+tl>IXSIZE) tl=IXSIZE-tx;
		if(tx<0)
		{
			tl+=tx;
			tx=0;
		}
		if(bitsperpixel==16)
		{
			put=(unsigned short *)(videomem+ty*stride+(tx<<1));
			while(tl-->0)
				*put++=color;
		} else
		{
			pc=videomem+ty*stride+tx;
			while(tl-->0)
				*pc++=color;
		}
	}
}

plotballs()
{
struct ball *aball;
int x,y;
int i,j;
int tx,ty;
int color;
int index;
int red,green,blue;
float bright;

	memset(vals,0,sizeof(vals));
	aball=activeballs;
	while(aball)
	{
		x=aball->x>>FRACTION;
		y=aball->y>>FRACTION;
		tx=x>>MIXFRAC;
		ty=y>>MIXFRAC;
		x=(float)x*IXSIZE/VIRTX;
		y=(float)y*IYSIZE/VIRTY;
		if(mixing)
		{
			if(tx>=0 && tx<(VIRTX>>MIXFRAC) && ty>=0 && ty<(VIRTY>>MIXFRAC))
			{
				index=ty*(VIRTX>>MIXFRAC)+tx;
				++vals[0][index];
				vals[1][index]+=aball->red;
				vals[2][index]+=aball->green;
				vals[3][index]+=aball->blue;
			}
			drawfigure(x,y,ballarray[makeindex(aball->red,aball->green,aball->blue)]);
//			drawsolid(x,y,disk,aball->red,aball->green,aball->blue);
		} else
			drawfigure(x,y,ballarray[aball->index]);
		aball=aball->next;
	}

	if(mixing)
	{
		for(i=0;i<(VIRTX>>MIXFRAC)*(VIRTY>>MIXFRAC);++i)
		{
			if(j=vals[0][i])
			{
				red=vals[1][i]/j;
				green=vals[2][i]/j;
				blue=vals[3][i]/j;
				bright=red>green ? red : green;
				bright=bright>blue ? bright : blue;
				if(bright)
				{
					bright=255/bright;
					vals[1][i]=red*bright;
					vals[2][i]=green*bright;
					vals[3][i]=blue*bright;
				}
			}
		}
		aball=activeballs;
		while(aball)
		{
			x=aball->x>>FRACTION;
			y=aball->y>>FRACTION;
			tx=x>>MIXFRAC;
			ty=y>>MIXFRAC;
			if(tx>=0 && tx<(VIRTX>>MIXFRAC) && ty>=0 && ty<(VIRTY>>MIXFRAC))
			{
				index=ty*(VIRTX>>MIXFRAC)+tx;
				aball->red=vals[1][index];
				aball->green=vals[2][index];
				aball->blue=vals[3][index];
			}
			aball=aball->next;
		}
	}
}

moveballs()
{
struct ball *aball;
float s,c,a,x,y,dx,dy,v;

	a=45*PI/180;
	s=sin(a);
	c=cos(a);
	aball=activeballs;
	while(aball)
	{
		bbmball(aball);

		aball->dy+=gravity;

		aball=aball->next;
	}
}

cullballs()
{
struct ball *aball, **aballptr;
	aballptr=&activeballs;
	while(*aballptr)
	{
		aball=*aballptr;
		if(aball->y>=VIRTY<<FRACTION)
		{
			*aballptr=aball->next;
			freeball(aball);
			continue;
		}
		aballptr=&aball->next;
	}
}


tritest()
{
float angle1,angle2,angle3;
int i;
int x1,y1,x2,y2,x3,y3;

#define CX (IXSIZE/2)
#define CY (IYSIZE/2)
#define R 192
	for(i=0;i<256;++i)
	{
		scrlock();
		angle1=i*PI/180;
		angle2=(i+120)*PI/180;
		angle3=(i-120)*PI/180;
		x1=CX+cos(angle1)*R;
		y1=CY+sin(angle1)*R;
		x2=CX+cos(angle2)*R;
		y2=CY+sin(angle2)*R;
		x3=CX+cos(angle3)*R;
		y3=CY+sin(angle3)*R;
		clear();
		tri(x1,y1,x2,y2,x3,y3,scanfunc,maprgb(0,0,0));
		scrunlock();
	}
	
}

onetri()
{
float angle1,angle2,angle3;
static int i;
int x1,y1,x2,y2,x3,y3;

	i+=1;
	angle1=i*PI/180;
	angle2=(i+120)*PI/180;
	angle3=(i-120)*PI/180;
	x1=CX+cos(angle1)*R;
	y1=CY+sin(angle1)*R;
	x2=CX+cos(angle2)*R;
	y2=CY+sin(angle2)*R;
	x3=CX+cos(angle3)*R;
	y3=CY+sin(angle3)*R;
	tri(CX,CY,x2,y2,x3,y3,scanfunc,maprgb(96,96,96));
	tri(x1,y1,CX,CY,x3,y3,scanfunc,maprgb(96,96,96));
/*
	x1=CX+cos(angle1)*R*2;
	y1=CY+sin(angle1)*R*2;
	x2=CX+cos(angle2)*R*2;
	y2=CY+sin(angle2)*R*2;
*/
	tri(x1,y1,x2,y2,CX,CY,scanfunc,maprgb(96,96,96));

}

minitri(int x,int y,int a)
{
int x1,y1,x2,y2,x3,y3;
int r;
float da,angle;
	r=9;
	da=PI*3/4;
	angle=a*PI/180;
	angle+=PI/2;
	x1=x+r*sin(angle);
	y1=y+r*cos(angle);
	x2=x+r*sin(angle-da);
	y2=y+r*cos(angle-da);
	x3=x+r*sin(angle+da);
	y3=y+r*cos(angle+da);
	tri(x1,y1,x2,y2,x,y,scanfunc,0);
	tri(x1,y1,x3,y3,x,y,scanfunc,0);
}


dots()
{
float angle,r,f;
int x,y;
int i,j,num,space;
int *points;
int *p1;
int t,t2;
int *angles,*a2;
int indexes[256];
static int endmark=0;

	for(i=0;i<256;++i)
		indexes[i]=makeindex(rand()&255,rand()&255,rand()&255);


	p1=points=malloc(1000000*sizeof(int));
	if(!points) nomem("dots");
	angles=a2=p1+500000;

	num=0;
	f=1.3;
	r=5;
	while(r<460)
	{
		x=r*sin(r/f)+IXSIZE/2;
		y=r*cos(r/f)+IYSIZE/2;
		*p1++=x;
		*p1++=y;
		*a2++=r/f*180/PI;
		++num;
		r=r+3/r;
	}
	i=0;
	space=4;
	while(!exitflag)
	{
		scaninput();
		scrlock();
		clear();
		j=endmark-space;
		t=0;
		while(j>=0)
		{
			y=j+j;
			x=points[y++];
			y=points[y];
			drawfigure(x,y,ballarray[indexes[t&255]]);
			++t;
			j-=space;
		}
		j=endmark<<1;
		minitri(points[j],points[j+1],angles[endmark]);
		++endmark;
		if(endmark>=num) endmark=num-space*7;
		scrunlock();
		++frames;
	}
	exitflag=0;
	dospiral=0;
	free(points);
}

#define MAXPRESETS 10

makepresetname(char *dest,int num)
{
char *p;
	p=getenv("HOME");
	if(p)
		sprintf(dest,"%s/.fountspreset%d",p,num);
	else
		sprintf(dest,"founts%d",num);
}

loadpresets()
{
int i;
int file,len;
char temp[256];
	for(i=0;i<MAXPRESETS;++i)
	{
		if(preset[i]) {free(preset[i]);preset[i]=0;}
		makepresetname(temp,i);
		file=open(temp,O_RDONLY);
		if(file>=0)
		{
			len=read(file,block64,65535);
			if(len>=0)
			{
				block64[len]=0;
				preset[i]=malloc(len+1);
				if(preset[i])
					strcpy(preset[i],block64);
			}
			close(file);
		}
	}
}


writepreset(unsigned int code)
{
struct hose *ahose;
struct block *ablock;
uchar *p;
char temp[256];
int file;

	if(code>=MAXPRESETS) return;

	makepresetname(temp,code);

	p=block64;
	ahose=activehoses;
	while(ahose)
	{
		sprintf(p,"hose:%d,%d,%d,%d,%d,%d\n",ahose->x,ahose->y,ahose->angle,
				ahose->red,ahose->green,ahose->blue);
		p+=strlen(p);
		ahose=ahose->next;
	}
	ablock=activeblocks;
	while(ablock)
	{
		sprintf(p,"block:%d,%d,%d\n",ablock->x,ablock->y,ablock->angle);
		p+=strlen(p);
		ablock=ablock->next;
	}
	*p++=0;
	if(preset[code]) free(preset[code]);
	preset[code]=malloc(p-block64);
	if(preset[code])
	{
		strcpy(preset[code],block64);
		file=open(temp,O_WRONLY|O_CREAT|O_TRUNC,0644);
		if(file>=0)
		{
			write(file,block64,p-block64-1);
			close(file);
		}
	}
}
deleteall()
{
struct hose *ahose,*ahose2;
struct block *ablock,*ablock2;
	ahose=activehoses;
	activehoses=0;
	while(ahose)
	{
		ahose2=ahose;
		ahose=ahose->next;
		freehose(ahose2);
	}
	ablock=activeblocks;
	activeblocks=0;
	while(ablock)
	{
		ablock2=ablock;
		ablock=ablock->next;
		freeblock(ablock2);
	}
	makebbm();
}

readpreset(int code)
{
struct hose *ahose,*lasthose;
struct block *ablock,*lastblock;
char *p;
int v1,v2,v3,v4,v5,v6;

	if(code>=MAXPRESETS) return;
	p=preset[code];
	if(!p) return;
	deleteall();
	lasthose=0;
	lastblock=0;
	while(*p)
	{
		if(sscanf(p,"hose:%d,%d,%d,%d,%d,%d",&v1,&v2,&v3,&v4,&v5,&v6)==6)
		{
			ahose=allochose();
			if(ahose)
			{
				ahose->x=v1;
				ahose->y=v2;
				ahose->angle=v3;
				ahose->red=v4;
				ahose->green=v5;
				ahose->blue=v6;
				ahose->index=makeindex(v4,v5,v6);
				ahose->next=0;
				if(lasthose)
					lasthose->next=ahose;
				else
					activehoses=ahose;
				lasthose=ahose;
			}
		} else
		if(sscanf(p,"block:%d,%d,%d",&v1,&v2,&v3)==3)
		{
			ablock=allocblock();
			if(ablock)
			{
				ablock->x=v1;
				ablock->y=v2;
				ablock->angle=v3;
				ablock->next=0;
				if(lastblock)
					lastblock->next=ablock;
				else
					activeblocks=ablock;
				lastblock=ablock;
			}
		}
		while(*p && *p++!='\n');
	}
	makebbm();
}

processkey(int key,int mod)
{
	if(key>=SDLK_1 && key<=SDLK_9)
	{
		key-=SDLK_1;
		if(mod&(KMOD_RSHIFT|KMOD_LSHIFT))
			writepreset(key);
		else
			readpreset(key);
		return;
	}
	switch(key)
	{
	case SDLK_m:
		mixing=!mixing;
		break;
	case SDLK_c:
		changefcolors();
		break;
	case SDLK_r:
		rotating=!rotating;
		break;
	case SDLK_s:
		dospiral=!dospiral;
		break;
	case SDLK_e:
		dosphere=!dosphere;
		break;
	case SDLK_f:
		fractal();
		break;
	case SDLK_h:
		dohide=!dohide;
		break;
	case SDLK_x:
		deleteall();
		break;
	case SDLK_p:
		pauseflag=!pauseflag;
		break;
	}
}

processmouse()
{
	hosehit();
	blockhit();
}

putcolors()
{
static unsigned short setup=0,cmap[512],*p;
int i,j,k,t,r,g,b,x,y,c;

	if(!setup)
	{
		t=0;
		for(k=0;k<8;++k)
		{
			r=k*36.43;
			for(j=0;j<8;++j)
			{
				g=j*36.43;
				for(i=0;i<8;++i)
				{
					b=i*36.43;
					cmap[t++]=maprgb(r,g,b);
				}
			}
		}
		setup=1;
	}
	t=0;
	for(k=0;k<8;++k)
	{
		for(j=0;j<8;++j)
		{
			for(i=0;i<8;++i)
			{
				c=cmap[t++];
				for(y=0;y<3;++y)
				{
					p=(unsigned short *)(videomem+
						stride*((k<<2)+((j&4)<<3)+y+100)+(100+(((j&3)<<5)+(i<<2)<<1)));
					for(x=0;x<3;++x)
						*p++=c;
				}
			}
		}
	}
}




doballs()
{
char redraw;
int selected;
int numballs;
int lasttime;
int newtime;
int i;

//tritest();
//return;
//dots();
	fractal();
	initballs();
	initcolors();
	inithoses();
	initblocks();
	initbbm();
	addhose(255,255,255);
	readpreset(0);
	pulseon();
	while(!exitflag)
	{
		scaninput();
		if(dospiral) {dots();hc=1;}
		if(dosphere) {sphere();hc=1;}

		scrlock();
		clear();
		plotballs();
		if(!dohide)
		{
			plothoses();
			plotblocks();
			drawcolors();
		}
		drawfigure(mousex*IXSIZE/VIRTX,mousey*IYSIZE/VIRTY,dummyball);
//putcolors();
		scrunlock();
		while(pauseflag&&!exitflag)
		{
			scaninput();
			hc=1;
		}
		++frames;

/*
This is the UNIX style of waiting, without wasting CPU cycles. Not
portable though

		if(!interrupted)
			pause();
		interrupted=0;
*/

		while(hc>0)
		{
			if(rotating) spinangle+=47;
			else spinangle=0;
			dohoses();
			moveballs();
			cullballs();
			--hc;
		}
	}
}

updatemap()
{
	SDL_SetColors(thescreen, themap, 0, 256);
}

int scrlock()
{
	if(SDL_MUSTLOCK(thescreen))
	{
		if ( SDL_LockSurface(thescreen) < 0 )
		{
			fprintf(stderr, "Couldn't lock display surface: %s\n",
								SDL_GetError());
			return -1;
		}
	}
	return 0;
}
scrunlock()
{
	if(SDL_MUSTLOCK(thescreen))
		SDL_UnlockSurface(thescreen);
	SDL_UpdateRect(thescreen, 0, 0, 0, 0);
}

drawsphere(int r,int bx,int by,int vx,int vy,int vz,int rt,int gt,int bt)
{
float x,y,y2,z,radius;
int i,j,j2,k,rint,rint2;
float tvx,tvy,tvz;
float dot;
unsigned short *p;
unsigned char *pc;
unsigned char red[256],green[256],blue[256];
float lvx,lvy,lvz;
int rr,gg,bb;
int row,col;

	for(i=0;i<256;++i)
	{
		if(i<232)
		{
			red[i]=224;
			green[i]=224;
			blue[i]=224;
		}
		else if(i<250)
		{
			red[i]=rt;
			green[i]=gt;
			blue[i]=bt;
		} else
			red[i]=green[i]=blue[i]=0;
	}

	lvx=-1.0;
	lvy=1.0;
	lvz=1.5;
	radius=sqrt(lvx*lvx+lvy*lvy+lvz*lvz);
	lvx/=radius;
	lvy/=radius;
	lvz/=radius;

	tvx=vx;
	tvy=vy;
	tvz=vz;
	radius=sqrt(tvx*tvx+tvy*tvy+tvz*tvz)/127.5;
	tvx/=radius;
	tvy/=radius;
	tvz/=radius;


	bx=bx*IXSIZE/640;
	by=by*IYSIZE/480;
	r=r*IXSIZE/640;

	radius=r;
	rint=r;
	rint2=rint*rint;
	row=(IYSIZE>>1)+by;
	col=(IXSIZE>>1)+bx-rint;
	for(j=-rint;j<=rint;++j)
	{
		p=(unsigned short *)(pc=videomem+stridemult[row-j])+col;
		pc+=col;
		y=j/radius;
		y2=y*y;
		j2=j*j;
		for(i=-rint;i<=rint;++i,++p)
		{
			if(i*i+j2>=rint2) continue;
			x=i/radius;
			z=sqrt(1.0-x*x-y2);
			k=127.5+x*tvx+y*tvy+z*tvz;
			dot=(x*lvx+y*lvy+z*lvz+1)/2;
			dot*=dot*dot;
			rr=red[k];
			gg=green[k];
			bb=blue[k];

			if(dot>=0.98)
			{
				rr=rr+3*255>>2;
				gg=gg+3*255>>2;
				bb=bb+3*255>>2;
			}

			rr*=dot;
			gg*=dot;
			bb*=dot;

			*p=maprgb(rr,gg,bb);
		}
	}
}
unsigned char rt[256],bt[256],gt[256];
initeyes()
{
int i;
	for(i=0;i<256;++i)
	{
		rt[i]=rand();
		gt[i]=rand();
		bt[i]=rand();
	}
}

sphere()
{
int i=0;
float a;
int x,y,z,u,v;
int s,c;
int j;
int k;

	while(!exitflag)
	{
		scaninput();
		scrlock();
		clear();

		x=mousex-VIRTX/2;
		y=VIRTY/2-mousey;
		z=60*5*IXSIZE/640;

		k=0;

/*
		drawsphere(120,-120,0,x,y,z,64,0,0);
		drawsphere(120,120,0,x,y,z,64,0,0);
*/
/*

		i=120;
		for(j=0;j<360;j+=i)
		{
			a=(j+(i>>1))*3.1415927/180;
			u=120*cos(a);
			v=120*sin(a);
			drawsphere(80,u,v,x-u,y+v,z,rt[k],gt[k],bt[k]);
			++k;
		}
*/

		for(j=0;j<7;++j)
		{
			v=(j-3)*40;
			for(i=0;i<7;++i)
			{
				u=(i-3)*40;
				drawsphere(20,u,v,x-u,y+v,z,rt[k],gt[k],bt[k]);
				++k;
			}
		}


		drawfigure(mousex*IXSIZE/VIRTX,mousey*IYSIZE/VIRTY,dummyball);
		scrunlock();
		++frames;
	}
	exitflag=0;
	dosphere=0;
}

#define ASQR 1025

unsigned char (*fractarr)[ASQR]=0;


diamond(int x,int y1,int y2,int y,int x1,int x2,int delta)
{
int c,average;
	if(x<0 || x>ASQR || y<0 || y>ASQR) return;
	c=average=0;
	if(y1>=0)
	{
		++c;
		average+=fractarr[x][y1];
	}
	if(y2<ASQR)
	{
		++c;
		average+=fractarr[x][y2];
	}
	if(x1>=0)
	{
		++c;
		average+=fractarr[x1][y];
	}
	if(x2<ASQR)
	{
		++c;
		average+=fractarr[x2][y];
	}
	average=average/c+rand()%delta-(delta>>1);
	if(average<0) average=0;
	if(average>255) average=255;
	fractarr[x][y]=average;
}
fcenter(int step,int delta)
{
int i,j;
int average;
int step2;
	step2=step>>1;

	for(j=0;j<ASQR-1;j+=step)
		for(i=0;i<ASQR-1;i+=step)
		{
			average=fractarr[i][j]+fractarr[i+step][j]+
				fractarr[i][j+step]+fractarr[i+step][j+step]>>2;
			average+=rand()%delta-(delta>>1);
			if(average<0) average=0;
			if(average>255) average=255;
			fractarr[i+step2][j+step2]=average;
		}
}
fdiamond(int step,int delta)
{
int i,j,average;
int step2;
	step2=step>>1;
	for(j=0;j<ASQR-1;j+=step)
		for(i=0;i<ASQR-1;i+=step)
		{
			diamond(i+step2,j-step2,j+step2,j,i,i+step,delta);
			diamond(i,j,j+step,j+step2,i-step2,i+step2,delta);
		}
}

int fmap=0;

copyfract()
{
unsigned char *p,*pb,*fbtake,*fbput;
unsigned short *ps;
int i,j,k;
int rmap[256],w;
int r,g,b;

	if(!fractarr) return;
	for(i=0;i<256;++i)
	{
		switch(fmap&15)
		{
		default:
			fmap=0;
		case 0:
			j=i<128 ? 0 : i-128<<1;
			r=j;
			g=j;
			b=j<128 ? 128 : j;
			break;
		case 1:
			if(i<64) r=g=b=0;
			else if(i<128) r=(i<<2)&255,g=0,b=0;
			else if(i<192) r=255,g=(i<<2)&255,b=0;
			else r=255,g=255,b=(i<<2)&255;
			break;
		case 2:
			if((i&15)<8) r=g=b=0;
			else r=g=b=255;
			break;
		case 3:
			r=g=b=((i&15)==8) ? 0 : 255;
			break;
		case 4:
			j=i>>5;
			r=(j&1) ? 255 : 0;
			g=(j&2) ? 255 : 0;
			b=(j&4) ? 255 : 0;
			break;
		case 5:
			j=i>>5;
			r=(j&1) ? 255 : 0;
			g=(j&2) ? 255 : 0;
			b=(j&4) ? 255 : 0;
			j=i&31;
			if(j>=16) j=31-j;
			r=r*j>>4;
			g=g*j>>4;
			b=b*j>>4;
			break;
		case 6:
			j=(i&63)<<4;
			if(i<64) r=0,g=0,b=64;
			else if(i<128) r=j,g=0,b=0;
			else if(i<192) r=0,g=j,b=0;
			else r=g=b=j;
			break;
		case 7:
			j=(i&63)<<2;
			if(i<64) r=0,g=0,b=64;
			else if(i<128) r=j,g=0,b=0;
			else if(i<192) r=0,g=j,b=0;
			else r=g=b=j;
			break;
		}
		rmap[i]=maprgb( r,g,b);
	}

	w=bitsperpixel==8 ? IXSIZE : IXSIZE<<1;
	if(!fractback)
		fractback=malloc(IYSIZE*w);
	if(!fractback)
		return;

	for(j=0;j<IYSIZE;++j)
	{
		p=fractarr[j];
		pb=fractback+j*w;
		ps=(unsigned short *)pb;
		for(i=0;i<IXSIZE;++i)
		{
			if(bitsperpixel==8)
				*pb++=rmap[*p++];
			else
				*ps++=rmap[*p++];
		}
	}
}


fractal()
{
int i,j,k;
unsigned char *p,*pb,*fbtake,*fbput;
unsigned short *ps;
int rmap[256],w;
int r,g,b;
	
	if(!fractarr)
		fractarr=malloc(ASQR*ASQR);
	if(!fractarr)
		return;

	memset(fractarr,128,ASQR*ASQR);
	j=ASQR-1;
	k=1024;
	while(j>1 && k>0)
	{
		fcenter(j,k);
		fdiamond(j,k);
		j>>=1;
		k>>=1;
	}
	copyfract();
}
changefcolors()
{
	++fmap;
	copyfract();
}


makeballs()
{
int i,j,k,t,r,g,b;
	t=0;
	for(i=0;i<8;++i)
	{
		r=i*36.42;
		for(j=0;j<8;++j)
		{
			g=j*36.42;
			for(k=0;k<8;++k)
			{
				b=k*36.42;
				ballarray[t++]=makeball(7,r,g,b);
			}
		}
	}
	dummyball=makeball(3,255,255,255);
}


main(int argc,char **argv)
{
int i,j,k,t;
long t1,t2;
int code;
char *p;
int starttime;

	initeyes();
	printf("Founts by David Ashley\n");
	printf("http://www.xdr.com/dash\n");
	printf("dash@xdr.com\n");
	printf("Use: founts [screenwidth]\n");
	printf("screenwidth is optional argument 100-2048 (default 640)\n");
	printf("c          Change background colors\n");
	printf("e          Eye demo\n");
	printf("f          New fractal background\n");
	printf("h          Hide/unhide the non-ball objects\n");
	printf("m          Turn on/off mixing\n");
	printf("p          Pause movement\n");
	printf("r          Rotate hoses\n");
	printf("s          Spiral pattern (escape to exit)\n");
	printf("x          Delete everything\n");
	printf("1-9        Load preset\n");
	printf("shift 1-9  Store preset\n");
	printf("Drag the fountains over the colors to change their color\n");

	if(argc>1)
	{
		i=atoi(argv[1]);
		if(i>=100 && i<=2048)
		{
			IYSIZE=IYSIZE*i/IXSIZE;
			IXSIZE=i;
		}
	}

	initmyrand();

	openx();
	block64=malloc(65536);
	if(!block64) nomem("block64");
	loadpresets();
	makeballs();

	mixing=0;
	dospiral=0;
	dosphere=0;
	dohide=0;
	rotating=0;

	framecount=0;
	gamemode=0;
	exitflag=0;
	starttime=SDL_GetTicks();
	frames=0;
	doballs();
	starttime=SDL_GetTicks()-starttime;
	if(!starttime) starttime=1;
	closex();
	printf("fps = %d\n",1000*frames/starttime);

}

drawfigure(int x,int y,unsigned short *take)
{
unsigned short *p1,*p2;
int run;
int dx,dy;
int xsize,ysize;
unsigned char *pc;

	if(!take) take=dummyball;

	dx=*((signed short *)take)++;
	dy=*((signed short *)take)++;
	xsize=*take++;
	ysize=*take++;
	x+=dx;
	y+=dy;
	if(x>=0 && y>=0 && x<=IXSIZE-xsize && y<=IYSIZE-ysize)
	{
		while(run=*take++)
		{
			dx=*((signed short *)take)++;
			dy=*((signed short *)take)++;
			if(bitsperpixel==16)
			{
				p1=(unsigned short *)(videomem+stridemult[y+dy]+(x+dx<<1));
				while(run--)
					*p1++=*take++;
			} else
			{
				pc=videomem+stridemult[y+dy]+x+dx;
				while(run--)
					*pc++=*take++;
			}
		}
	} else
	{
		while(run=*take++)
		{
			dx=*((signed short *)take)++;
			dy=*((signed short *)take)++;
			dx+=x;
			dy+=y;
			p2=take;
			take+=run;
			if(dy<0 || dy>=IYSIZE) continue;
			if(dx>=IXSIZE) continue;
			if(dx<0)
			{
				p2-=dx;
				run+=dx;
				dx=0;
			} else if(dx+run>IXSIZE)
				run=IXSIZE-dx;
			if(bitsperpixel==16)
			{
				p1=(unsigned short *)(videomem+stridemult[dy]+(dx<<1));
				while(run-->0)
					*p1++=*p2++;
			} else
			{
				pc=videomem+stridemult[dy]+dx;
				while(run-->0)
					*pc++=*p2++;
			}
		}
	}

}
closex()
{
	if(sdlloaded)
		SDL_Quit();
}
openx()
{
unsigned long videoflags;
int i;

	for(i=0;i<256;++i)
	{
		themap[i].r=i;
		themap[i].g=i;
		themap[i].b=i;

	}
	if ( SDL_Init(SDL_INIT_VIDEO|SDL_INIT_TIMER) < 0 )
	{
		fprintf(stderr, "Couldn't initialize SDL: %s\n",SDL_GetError());
		exit(1);
	}
	videoflags = SDL_SWSURFACE|SDL_HWPALETTE|SDL_ANYFORMAT;

	thescreen = SDL_SetVideoMode(IXSIZE, IYSIZE, 16, videoflags);
	if ( thescreen == NULL )
	{
		fprintf(stderr, "Couldn't set display mode: %s\n",
							SDL_GetError());
		SDL_Quit();
		exit(5);
	}
	sdlloaded=1;
	stride=thescreen->pitch;
	videomem=thescreen->pixels;
	bitsperpixel=thescreen->format->BitsPerPixel;
	for(i=0;i<IYSIZE;++i) stridemult[i]=i*stride;
	updatemap();
	SDL_ShowCursor(0);
}
scaninput()
{
SDL_Event event;
int key,mod;

	while(SDL_PollEvent(&event))
	{
		switch(event.type)
		{
		case SDL_KEYDOWN:
			key=event.key.keysym.sym;
			mod=event.key.keysym.mod;
 			if(key==SDLK_ESCAPE)
				exitflag=1;
			else processkey(key,mod);
			break;
		case SDL_MOUSEBUTTONUP:
			buttonstate=NOTDOWN;
			break;
		case SDL_MOUSEBUTTONDOWN:
			mousex=event.button.x*VIRTX/IXSIZE;
			mousey=event.button.y*VIRTY/IYSIZE;
			buttonstate=JUSTDOWN;
			processmouse();
			break;
		case SDL_MOUSEMOTION:
			mousex=event.motion.x*VIRTX/IXSIZE;
			mousey=event.motion.y*VIRTY/IYSIZE;
			if(buttonstate==JUSTDOWN) buttonstate=CONTDOWN;
			processmouse();
			break;
		}
	}
}
clear()
{
unsigned char *p1,*p2;
int i,w;

	p1=videomem;
	p2=fractback;
	w=(bitsperpixel==16) ? IXSIZE<<1 : IXSIZE;
	for(i=0;i<IYSIZE;++i)
	{
		if(!p2)
			memset(p1,0,w);
		else
		{
			memmove(p1,p2,w);
			p2+=w;
		}
		p1+=stride;
	}
}

#define TRIFRAC 16
#define HALF (1<<TRIFRAC-1)

void tri(int x1,int y1,int x2,int y2,int x3,int y3,
		void (*func)(int,int,int,long),long color)
{
int xc1,xc2,xc3,xd1,xd2,dy;
int t;

	if(y1>y2) {t=x1;x1=x2;x2=t;t=y1;y1=y2;y2=t;}
	if(y1>y3) {t=x1;x1=x3;x3=t;t=y1;y1=y3;y3=t;}
	if(y2>y3) {t=x2;x2=x3;x3=t;t=y2;y2=y3;y3=t;}
	xc1=x1<<TRIFRAC;
	xc3=x3<<TRIFRAC;
	dy=y3-y1-1;
	if(dy<2) dy=2;
	xd1=(xc3-xc1)/dy;
	if(y2==y1)
	{
		xc2=x2<<TRIFRAC;
		xd2=(xc3-xc2)/dy;
	} else
	{
		xc2=xc1;
		dy=y2-y1-1;
		if(dy<2) dy=2;
		xd2=((x2<<TRIFRAC)-xc2)/dy;
	}
	while(y1<y3)
	{
		func(xc1+HALF>>TRIFRAC,y1,xc2+HALF>>TRIFRAC,color);
		xc1+=xd1;
		xc2+=xd2;
		++y1;
		if(y1==y2)
		{
			xc2=x2<<TRIFRAC;
			dy=y3-y2-1;
			if(dy<2) dy=2;
			xd2=(xc3-xc2)/dy;
		}
	}
}
