
//**************************************************************************
//**
//** i_cdmus.c
//**
//**************************************************************************

// HEADER FILES ------------------------------------------------------------

#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include "h2def.h"
#include "i_sound.h"
#include "SDL_cdrom.h"

// EXTERNAL DATA DECLARATIONS ----------------------------------------------

// PUBLIC DATA DEFINITIONS -------------------------------------------------

int cd_Error;

// PRIVATE DATA DEFINITIONS ------------------------------------------------

static SDL_CD *sdl_cdrom;

// CODE --------------------------------------------------------------------

//==========================================================================
//
// I_CDMusInit
//
// Initializes the CD audio system.  Must be called before using any
// other I_CDMus functions.
//
// Returns: 0 (ok) or -1 (error, in cd_Error).
//
//==========================================================================

int I_CDMusInit(void)
{
	cd_Error = 0;
	if ( SDL_CDNumDrives() > 0 ) {
		if ( (sdl_cdrom=SDL_CDOpen(0)) == NULL ) {
			cd_Error = CDERR_NOTINSTALLED;
		}
	} else {
		cd_Error = CDERR_NOTINSTALLED;
	}
	if ( cd_Error ) {
		return(-1);
	} else {
		return(0);
	}
}

//==========================================================================
//
// I_CDMusPlay
//
// Play an audio CD track.
//
// Returns: 0 (ok) or -1 (error, in cd_Error).
//
//==========================================================================

int I_CDMusPlay(int track)
{
	cd_Error = 0;
	if ( CD_INDRIVE(SDL_CDStatus(sdl_cdrom)) ) {
		if ( SDL_CDPlayTracks(sdl_cdrom, track, 0, 1, 0) < 0 ) {
			cd_Error = CDERR_BADTRACK;
		}
	} else {
		cd_Error = CDERR_NOTINSTALLED;
	}
	if ( cd_Error ) {
		return(-1);
	} else {
		return(0);
	}
}

//==========================================================================
//
// I_CDMusStop
//
// Stops the playing of an audio CD.
//
// Returns: 0 (ok) or -1 (error, in cd_Error).
//
//==========================================================================

int I_CDMusStop(void)
{
	if ( SDL_CDPause(sdl_cdrom) < 0 ) {
		cd_Error = CDERR_NOTINSTALLED;
		return(-1);
	}
	return(0);
}

//==========================================================================
//
// I_CDMusResume
//
// Resumes the playing of an audio CD.
//
// Returns: 0 (ok) or -1 (error, in cd_Error).
//
//==========================================================================

int I_CDMusResume(void)
{
	if ( SDL_CDResume(sdl_cdrom) < 0 ) {
		cd_Error = CDERR_NOTINSTALLED;
		return(-1);
	}
	return(0);
}

//==========================================================================
//
// I_CDMusSetVolume
//
// Sets the CD audio volume (0 - 255).
//
// Returns: 0 (ok) or -1 (error, in cd_Error).
//
//==========================================================================

int I_CDMusSetVolume(int volume)
{
	return 0;
}

//==========================================================================
//
// I_CDMusFirstTrack
//
// Returns: the number of the first track.
//
//==========================================================================

int I_CDMusFirstTrack(void)
{
	int i, cd_FirstTrack = 0;

	if ( CD_INDRIVE(SDL_CDStatus(sdl_cdrom)) ) {
		for ( i=0; i<sdl_cdrom->numtracks; ++i ) {
			if ( sdl_cdrom->track[i].type == SDL_AUDIO_TRACK ) {
				cd_FirstTrack = i;
				break;
			}
		}
	}
	return cd_FirstTrack;
}

//==========================================================================
//
// I_CDMusLastTrack
//
// Returns: the number of the last track.
//
//==========================================================================

int I_CDMusLastTrack(void)
{
	int i, cd_LastTrack = 0;

	if ( CD_INDRIVE(SDL_CDStatus(sdl_cdrom)) ) {
		for ( i=sdl_cdrom->numtracks-1; i >= 0; --i ) {
			if ( sdl_cdrom->track[i].type == SDL_AUDIO_TRACK ) {
				cd_LastTrack = i;
				break;
			}
		}
	}
	return cd_LastTrack;
}

//==========================================================================
//
// I_CDMusTrackLength
//
// Returns: Length of the given track in seconds, or -1 (error, in
// cd_Error).
//
//==========================================================================

int I_CDMusTrackLength(int track)
{
	int length;

	if(track < I_CDMusFirstTrack() || track > I_CDMusLastTrack())
	{
		cd_Error = CDERR_BADTRACK;
		return -1;
	}
	return (sdl_cdrom->track[track].length/CD_FPS);
}
