/*
	File: 
		bmpfile.cpp
	Content: 
		simple BMP file loader. Only 24bit uncompressed DIB files are supported
	Author:
		Damyan Ognyanoff
		e-mail: damyan@rocketmail.com
	Remarks:

*/
#include <stdio.h>
#include <SDL.h>
#include "bmpfile.h"

// constructor:
//	number of images in the file	
// params: count - default 1
CBMPFile::CBMPFile(int count /*= 1*/)
{
	m_count = count;
	m_pBits = NULL;
} // constructor

// destructor
CBMPFile::~CBMPFile()
{
	if (m_pBits)
	{
		delete m_pBits;
		m_pBits = NULL;
	}
} // destructor

/*
Method:
	load
params:
	name of the file to load
return:
	false on error, true otherwise
 */
bool CBMPFile::load(char* pFileName)
{
	// open the file in read only, binary mode 
	FILE* fp = fopen( pFileName, "rb");
	// if an error occur exit forth
	if (fp == NULL) return false;

	BITMAPFILEHEADER bmpFH;
	// read the file header
	int bytesread = fread(&bmpFH, 1,sizeof(bmpFH), fp);
	// check for the size of the readed chunk
	if (bytesread == sizeof(bmpFH))
	{
		// check the first 2 bytes from the file to the signature 'BM'
		if (bmpFH.bfType == 0x4d42)
		{
			// it's ok - allocate enought space from the heap
			m_pBits = new unsigned char[ bmpFH.bfSize - sizeof(BITMAPFILEHEADER)];
			PBITMAPINFOHEADER pBmpInfo = (PBITMAPINFOHEADER)m_pBits;
			// read the rest of the info header
			bytesread = fread(m_pBits,  1,sizeof(BITMAPINFOHEADER), fp);
			// check the new chunk again
			if (bytesread == sizeof(BITMAPINFOHEADER))
			{
				// calc the rest of the bytes from the file
				int bytesToread = bmpFH.bfSize - sizeof(BITMAPFILEHEADER) - bytesread; 
				// read the image bits
				bytesread = fread(m_pBits+sizeof(BITMAPINFOHEADER), 1, bytesToread,  fp);
				// is all ok?
				if (bytesread == bytesToread)
				{
					// do nothing
				} else {
					// on error delete the allocated memory and return false
					delete m_pBits;
					m_pBits = NULL;
				}
			} else {
				// on error delete the allocated memory and return false
				delete m_pBits;
				m_pBits = NULL;
			} // read header
		} // if signature match 'BM'
	} // file header is processed
	// close the file
	fclose(fp);
	//return true to indicate that all is ok
	return true;
} // load()

/*
method:
	getHeader() - retrive pointer to the image header into the memory
return:
	pointer to bitmap info header
*/
PBITMAPINFOHEADER CBMPFile::getHeader()
{
	return (PBITMAPINFOHEADER)m_pBits;
} // getHeader()

/*
method:
	getBits() - retrive the pointer to acxtual image bits into the memory
return:
	pointer to the image bits
*/
Uint8* CBMPFile::getBits()
{	
	// they reside right after the Image header
	// NOTE: this is true only for the 24bit DIB images without the colorinfo in them
	// (e.g. pallete)
	return m_pBits+sizeof(BITMAPINFOHEADER);
} // getBits()

/*
method:
	getWidth() - retrive the width member of the bitmap info structure
return:
	width of the image
*/
int CBMPFile::getWidth()
{
	return getHeader()->biWidth;
} // getWidth()

/*
method:
	getHeight() - retrive the height member of the bitmap info structure
return:
	height of the image
*/
int CBMPFile::getHeight()
{
	return getHeader()->biHeight;
} // getHeight()


/*
class CImage
this class uses the main bitmap file represening the part of it as separate image
the original file is divided to several images vertically and the width of the 
image is the same.
*/

// constructor
// only pointer to main image is used
CImage::CImage(CBMPFile* pBitmap)
{
	m_pBitmap = pBitmap;
} // contructor

/*
method:
	getBits() - calculate and retrive the bits of the subimage into the main image using the index
param:
	index of the subimage to retrive
return:
	pointer to the bits of the subimage
*/
Uint8* CImage::getBits(int index)
{
	// check the index passed to be valid
	if (index >= m_pBitmap->m_count) return NULL;
	// retrive the height of the image
	int dy = getHeight();
	// each line is place on the 4 byte boundary, so calculate it properly
	int bytesperline = (m_pBitmap->getWidth()*3+3)&0xfffffffc;
	// compute
	return m_pBitmap->getBits()+dy*index*bytesperline;
} // getBits()

// get the widht of the subimage
int CImage::getWidth()
{
	// get it from the main image
	return m_pBitmap->getWidth();
} // getWidth()

// get the height of the image
int CImage::getHeight()
{
	// actually the height of the main image divided by the number of subimages on it
	return m_pBitmap->getHeight() / m_pBitmap->m_count;
} // getHeight()

// here is a small paint routine. it uses the Image pane with 24bit depth and places
// the image on x,y coords acording to the actual dimensions of the image pane
//	pBase - 
//		pointer to the memory image of the pane where the subimage will be painted
//	mapwidth, mapHeight - 
//		dimensions of the image pane
//	x, y - 
//		coords where we like to place the subimage
//	index - 
//		index of the subimage to paint
void CImage::paint(unsigned char* pBase, int mapWidth, int mapHeight, int x, int y, int index)
{
	// get image dimensions
	int h = getHeight();
	int w = getWidth();
	// calc the bytes per line in the image buffer
	int wperline = (w*3+3)&0xFFFFFFFC;
	Uint8* p = pBase;
	//for each line on the image
	for (int i = 0; i < h; i++)
	{
		// check that the line is 'visible' e.g. do not step outside of the pane
		if ((y+i) >= mapHeight) break;
		// for each pixel in the line
		for (int j = 0; j < w; j++)
		{
			// another check for pane boundary vilations
			if ((x+j) >= mapWidth) break;
			// calc the destination 
			RGBQUAD* dst = (RGBQUAD*)(p+((y+i)*mapWidth+x+j)*3);
			// calc the source 
			RGBQUAD* src = (RGBQUAD*)(getBits(index)+ ((h-i-1)*wperline)+j*3);
			// place the R,G and B components there
#ifdef ANONSTRUCT_NOT_SUPPORTED
			dst->st1.Red = src->st1.Red;
			dst->st1.Green = src->st1.Green;
			dst->st1.Blue = src->st1.Blue;
#else
			dst->st1.Red = src->st1.Red;
			dst->st1.Green = src->st1.Green;
			dst->st1.Blue = src->st1.Blue;
#endif
		} // for(horisontal)
	}// for (vertical)
} // paint()